/* global describe it before ethers */

import { expect } from 'chai';
import { ethers } from 'hardhat';
import { BigNumber, Signer } from 'ethers';
// Import contract types
import {
  TradingManagementStorageFacet,
  TradingManagementExecutorFacet,
  DiamondLoupeFacet,
} from '../../types';

// Add this import for the matcher

// Import Diamond utilities
import {
  deployDiamond,
  DiamondAddress,
} from '../../scripts/deployTradingManagement';
import {
  deployContractsForTests,
  FameContracts,
  getSigners,
} from './helpers/deploy-helper';
import { initializeContracts } from './helpers/contracts-initialization-helper';
import { Signers } from './types';
import { getSelectors } from '../../scripts/libraries/diamond';

describe('37 - TradingManagementExecutorFacet Integration Test', function () {
  // Contract instances
  let tradingManagementStorageFacet: TradingManagementStorageFacet;
  let tradingManagementExecutorFacet: TradingManagementExecutorFacet;
  let contracts: FameContracts;
  let signers: Signers;
  let diamondLoupeFacet: DiamondLoupeFacet;

  // Signers
  let owner: Signer;
  let customer: Signer;
  let dataProvider: Signer;

  // Constants
  const assetid = 'aabc';
  const oid = 'oabc';
  const assetPrice = ethers.utils.parseUnits('100', 18); // 100 payment tokens
  const initialPaymentTokenBalance = ethers.utils.parseUnits('1000', 18);

  before(async function () {
    // Get signers
    signers = await getSigners();
    owner = signers.admin;
    customer = signers.signer1;
    dataProvider = signers.signer2;

    // Deploy the Diamond contract
    const tradingManagementAddress = await deployDiamond({
      saveInfo: false,
      showInfo: false,
    });
    // Get instances of the core facets using the Diamond's address
    diamondLoupeFacet = await ethers.getContractAt(
      'DiamondLoupeFacet',
      DiamondAddress,
    );
    tradingManagementStorageFacet = await ethers.getContractAt(
      'TradingManagementStorageFacet',
      DiamondAddress,
    );
    tradingManagementExecutorFacet = await ethers.getContractAt(
      'TradingManagementExecutorFacet',
      DiamondAddress,
    );

    contracts = await deployContractsForTests({ shouldResetNetwork: false });
    await initializeContracts({ contracts, signers });

    // Set storage variables in the TradingManagementStorageFacet
    await tradingManagementStorageFacet
      .connect(owner)
      .setOfferingTokenContract(contracts.offeringTokenContract.address);
    await tradingManagementStorageFacet
      .connect(owner)
      .setPaymentTokenContract(contracts.paymentTokenContract.address);
    await tradingManagementStorageFacet
      .connect(owner)
      .setBourseContractAddress(contracts.bourseContract.address);
    await tradingManagementStorageFacet
      .connect(owner)
      .setPaygDataAccessContract(
        contracts.dataAccessPayAsYouGoContract.address,
      );
    await tradingManagementStorageFacet
      .connect(owner)
      .setPayuDataAccessContract(
        contracts.dataAccessPayAsYouUseContract.address,
      );
    await tradingManagementStorageFacet
      .connect(owner)
      .setSubscriptionDataAccessContract(
        contracts.dataAccessSubscriptionContract.address,
      );

    // Mint payment tokens to the customer
    await contracts.governanceContract
      .connect(owner)
      .mintCoin(await customer.getAddress(), 'FDE', initialPaymentTokenBalance);

    // Mint an asset to the data provider in the OfferingToken contract
    const result = await contracts.offeringTokenContract
      .connect(owner)
      .addAsset(
        assetid,
        oid,
        'https://example.com/token-uri-3',
        await dataProvider.getAddress(),
        assetPrice,
        1234567890,
        '10',
        '100MB',
        'target1',
        'sl1',
        '0x00',
      );

    await result.wait();
    expect(result).to.be.ok;
  });

  describe('Initialization tests', function () {
    it('should deploy TradingManagementDiamond correctly', async function () {
      expect(DiamondAddress).to.properAddress;
    });

    it('should have TradingManagementStorageFacet added to the Diamond', async function () {
      const diamondLoupeFacet = await ethers.getContractAt(
        'DiamondLoupeFacet',
        DiamondAddress,
      );
      const facetAddresses = await diamondLoupeFacet.facetAddresses();
      const storageFacetSelectors = getSelectors(tradingManagementStorageFacet);
      let storageFacetAddress: string | undefined;

      for (const address of facetAddresses) {
        const selectors =
          await diamondLoupeFacet.facetFunctionSelectors(address);
        if (selectors.some((s: string) => storageFacetSelectors.includes(s))) {
          storageFacetAddress = address;
          break;
        }
      }

      expect(storageFacetAddress).to.be.properAddress;
    });

    it('should have TradingManagementExecutorFacet added to the Diamond', async function () {
      const diamondLoupeFacet = await ethers.getContractAt(
        'DiamondLoupeFacet',
        DiamondAddress,
      );
      const facetAddresses = await diamondLoupeFacet.facetAddresses();
      const executorFacetSelectors = getSelectors(
        tradingManagementExecutorFacet,
      );
      let executorFacetAddress: string | undefined;

      for (const address of facetAddresses) {
        const selectors =
          await diamondLoupeFacet.facetFunctionSelectors(address);
        if (selectors.some((s: string) => executorFacetSelectors.includes(s))) {
          executorFacetAddress = address;
          break;
        }
      }

      expect(executorFacetAddress).to.be.properAddress;
    });

    it('should set OfferingTokenContract address correctly in storage', async function () {
      const storedAddress =
        await tradingManagementStorageFacet.getOfferingTokenContract();
      expect(storedAddress).to.equal(contracts.offeringTokenContract.address);
    });

    it('should set PaymentTokenContract address correctly in storage', async function () {
      const storedAddress =
        await tradingManagementStorageFacet.getPaymentTokenContract();
      expect(storedAddress).to.equal(contracts.paymentTokenContract.address);
    });

    it('should set BourseContractAddress correctly in storage', async function () {
      const storedAddress =
        await tradingManagementStorageFacet.getBourseContractAddress();
      expect(storedAddress).to.equal(contracts.bourseContract.address);
    });

    it('should set PaygDataAccessContract address correctly in storage', async function () {
      const storedAddress =
        await tradingManagementStorageFacet.getPaygDataAccessContract();
      expect(storedAddress).to.equal(
        contracts.dataAccessPayAsYouGoContract.address,
      );
    });

    it('should set PayuDataAccessContract address correctly in storage', async function () {
      const storedAddress =
        await tradingManagementStorageFacet.getPayuDataAccessContract();
      expect(storedAddress).to.equal(
        contracts.dataAccessPayAsYouUseContract.address,
      );
    });

    it('should set SubscriptionDataAccessContract address correctly in storage', async function () {
      const storedAddress =
        await tradingManagementStorageFacet.getSubscriptionDataAccessContract();
      expect(storedAddress).to.equal(
        contracts.dataAccessSubscriptionContract.address,
      );
    });

    it('should have customer with correct initial PaymentToken balance', async function () {
      const balance = await contracts.paymentTokenContract.balanceOf(
        await customer.getAddress(),
      );
      expect(balance).to.equal(initialPaymentTokenBalance);
    });

    it('should have customer with 0 initial data access token balances', async function () {
      const offeringTokenIndx =
        await contracts.offeringTokenContract.getIDHash(oid);
      const subscriptionBalance =
        await contracts.dataAccessSubscriptionContract.balanceOf(
          await customer.getAddress(),
          offeringTokenIndx,
        );

      const payAsYouGoBalance =
        await contracts.dataAccessPayAsYouGoContract.balanceOf(
          await customer.getAddress(),
          offeringTokenIndx,
        );

      const payAsYouUseBalance =
        await contracts.dataAccessPayAsYouUseContract.balanceOf(
          await customer.getAddress(),
          offeringTokenIndx,
        );

      expect(payAsYouGoBalance).to.equal(0);
      expect(subscriptionBalance).to.equal(0);
      expect(payAsYouUseBalance).to.equal(0);
    });

    it('should have dataProvider with 0 initial PaymentToken balance', async function () {
      const balance = await contracts.paymentTokenContract.balanceOf(
        await dataProvider.getAddress(),
      );
      expect(balance).to.equal(BigNumber.from(0));
    });

    it('should have asset added to OfferingToken contract', async function () {
      const exists = await contracts.offeringTokenContract.tokenExists(oid);
      expect(exists).to.be.true;
    });
  });
});
